using System;
using System.IO;
using System.Text;
using System.Xml;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class is a static module containing shared implementation units
	/// for logon response success/partialSuccess messages. 
	/// It is used to model/fake multiple implementation inheritance. 
	/// </summary>
	internal sealed class SecurityLogonNonFailureResponseMessageImplementationMixin
	{
		// Constants used in XML message serialization/deserialization
		private const string XMLCONSTS_POST_SIGN_IN_TEXT_NODE_NAME = "PostSignInText";
		private const string XMLCONSTS_LINE_NODE_NAME = "Line";

		private SecurityLogonNonFailureResponseMessageImplementationMixin() {}

		/// <summary>
		/// Logon non-failure messages common serialization code. It should be called 
		/// from corresponding XML serialization methods of non-failure logon response message classes. 
		/// </summary>
		/// <param name="writer">XmlWriter to use. Required.</param>
		/// <param name="postSignInText">Post sign-in text. It can be null.</param>
		/// <param name="forceWriteOfElement">Specifies whether the method should write out an element if the text is null.</param>
		internal static void WritePostSignInTextToXml( XmlWriter writer, string postSignInText, bool forceWriteOfElement )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			if( postSignInText == null || postSignInText == String.Empty )
			{
				if( forceWriteOfElement )
					writer.WriteElementString( XMLCONSTS_POST_SIGN_IN_TEXT_NODE_NAME, null );

				return;
			}
			
			StringReader _rdr = new StringReader( postSignInText );
			string _singleLine;

			writer.WriteStartElement( XMLCONSTS_POST_SIGN_IN_TEXT_NODE_NAME );

			while( (_singleLine = _rdr.ReadLine()) != null )
				writer.WriteElementString( XMLCONSTS_LINE_NODE_NAME, _singleLine );
				
			writer.WriteEndElement();
		}

		/// <summary>
		/// This method checks if response element's first child is the post sign-in text element,
		/// reads lines of text and returns concatenated result. It will throw an exception if
		/// if the post sign-in text element cannot be found and return null if it contains 
		/// no lines of text. It should be called from XML deserialization constructors 
		/// of non-failure logon response message classes. 
		/// </summary>
		/// <param name="responseElement">Message response element. Required.</param>
		/// <returns>Post sign-in text if it was found. Null otherwise.</returns>
		/// <param name="requirePostSignInTextElement">
		///		Specifies if post sign-in text element is required. 
		///		If it is then exception will be thrown if element is missing. Otherwise the method will return null.</param>
		internal static string ParsePostSignInTextXml( XmlElement responseElement, bool requirePostSignInTextElement )
		{
			if( responseElement == null )
				throw( new ArgumentNullException( "responseElement" ) );			

			if( responseElement.FirstChild == null || responseElement.FirstChild.Name != XMLCONSTS_POST_SIGN_IN_TEXT_NODE_NAME )
			{
				if( !requirePostSignInTextElement )
					return null;

				throw( new XmlParseException( SR.Exceptions.XmlParseRequiredElementNotFound( XMLCONSTS_POST_SIGN_IN_TEXT_NODE_NAME ) ) );
			}

			XmlNode _currentLine = responseElement.FirstChild.FirstChild;

			if( _currentLine == null )
				return null;

			StringBuilder _sb = new StringBuilder();
			bool _firstLineFlag = true;

			while( _currentLine != null )
			{
				if( _currentLine.NodeType != XmlNodeType.Element )
					continue;

				if( _currentLine.Name != XMLCONSTS_LINE_NODE_NAME )
					throw( new XmlParseException( SR.Exceptions.XmlParseRequiredElementNotFound( XMLCONSTS_LINE_NODE_NAME ) ) );

				if( !_firstLineFlag )
					_sb.Append( Environment.NewLine );
				else
					_firstLineFlag = false;

				_sb.Append( _currentLine.InnerText );

				_currentLine = _currentLine.NextSibling;
			}

			return _sb.ToString();
		}
	}
}
